
  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Tue Nov 25 10:17:10 2007
   *  Last modified   :  %modify_time%
   */
  /** @file 
   *  \brief A brief description of this module 
   *
   *  \version CRYS_ATP_TST_DESFLOW.c#1:csrc:1
   *  \author Annag
   *  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
   *           All Rights reserved
   */


/************* Include Files ****************/
#include "CRYS.h"

#include "CRYS_ATP_TST_Defines.h"
#include "CRYS_ATP_TST_UTIL_Funcs.h"
#include "CRYS_ATP_TST_UTIL_OTF_Funcs.h"
#include "CRYS_ATP_TST_DESFLOW_Acceptance.h"
#include "CRYS_ATP_TST_AESFLOW_Acceptance.h"
#include "CRYS_ATP_TST_HASHFLOW_Acceptance.h"
#include "CRYS_ATP_TST_DESFLOW_Acceptance_test_vectors.h"
#include "CRYS_ATP_TST_FLOW_Defines.h"
#include "gen.h"
#include "CRYS_INIT_TST.h"
#include "CRYS_FLOW.h"
#include "CRYS_FLOW_error.h"
#include "DX_VOS_Mem.h"
#include "FLOW_error.h"

#if TST_USE_FLOW_DATA

/************************ Defines *******************************/
/************************ Enums *********************************/
/************************ Typedefs ******************************/
/************************ Global Data ***************************/

extern DxUint8_t 	                       TST_Input_Data[TST_FLOW_DATA_VEC_SIZE_IN_BYTS];

extern DxUint64_t	                       TST_FLOW_InputData[TST_FLOW_DATA_VEC_SIZE_IN_BYTS/8];

extern DxUint64_t						   TST_FLOW_OutputData[TST_FLOW_NUMBER_OF_SUPPORTED_ENGINES][TST_FLOW_DATA_VEC_SIZE_IN_BYTS/8];

extern DxUint64_t	                       TST_FLOW_OutputDataForIntegratedEngines[TST_FLOW_DATA_VEC_SIZE_IN_BYTS/8];

/* common data for one test vector */
extern TST_FLOW_TestCommonDataStruct       TST_TestData;

/* input data size */
extern DxUint32_t                          TST_FLOW_InputDataSize;

extern CRYS_FLOW_Params_t                  TST_FLOWParamsInfo;

/* flow input buffers */
extern CRYS_FLOW_BufferData_t              TST_InputBuffers[TST_FLOW_DATA_MAX_NUM_OF_FLOWBUFFERS];

/* flow output buffers */
extern CRYS_FLOW_BufferData_t              TST_OutputBuffers[TST_FLOW_NUMBER_OF_SUPPORTED_ENGINES][TST_FLOW_DATA_MAX_NUM_OF_FLOWBUFFERS];

/* output data of HASH/AES_MAC result */
extern DxUint32_t                          TST_FLOW_Hash_OutputData[CRYS_HASH_RESULT_SIZE_IN_WORDS];

/* indication if data size in function finish should be zero */
extern DxBool_t                            TST_FLOW_IsFinishSizeZero;

CRYS_FLOW_DESEngineData_t                  TST_FLOWDESEngineData;


 
/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_InitDESEngineData
 *  
 * Inputs:
 *  TST_index - test index
 *  None
 *
 * Outputs:                                 
 *
 * Description:                                                                 
 *  This function initializes engine data for DES FLOW operation
 * Algorithm:     
 *	This function initializes DES flow engine data struct with values from the test vector.
 ******************************************************************************/                                
void CRYS_ATP_TST_FLOW_InitDESEngineData ( int TST_index ) 
{
   
    /* Init DES engine data */
    /************************/
    
    /* initialize IV counter */
    DX_VOS_FastMemCpy(TST_FLOWDESEngineData.IV_ptr, 
                      TST_DESFLOW_Acceptance_TestData[TST_index].TST_IvCounter,
                      CRYS_DES_IV_SIZE_IN_BYTES);

	/* number of keys */
	TST_FLOWDESEngineData.NumOfKeys = TST_DESFLOW_Acceptance_TestData[TST_index].TST_NumOfKeys;


    /* key data */                     
    TST_FLOWDESEngineData.Key_ptr = &TST_DESFLOW_Acceptance_TestData[TST_index].TST_Key;                      
                      
    /* operation mode */                      
	TST_FLOWDESEngineData.OperationMode = TST_DESFLOW_Acceptance_TestData[TST_index].TST_OperationMode;
     
}

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_InitDESFlow
 *  
 * Inputs:
 *  TST_index          - test index
 *  flowType           - flow type (static or dynamic)
 *  TST_MasterTestName - test name
 *  TST_DESFlowId      - pointer to fill with flow id
 *  pDESFlowParamsInfo - flow params info
 *  encryptMode        - encrypt/decrypt mode
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *	This function inits DES flow.
 *
 * Algorithm:     
 *	This function starts DES flow.
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_InitDESFlow ( int                     TST_index,
                                       CRYS_FLOW_Type_t        flowType,
                                       char                   *TST_MasterTestName, 
                                       DxUint32_t             *TST_DESFlowId,
                                       CRYS_FLOW_Params_t     *pDESFlowParamsInfo,
                                       CRYS_DES_EncryptMode_t  encryptMode ) 
{
    /********* TST Variables Definitions ************/
    CRYSError_t  TST_TestStatus = TESTPASS;
    
    TST_TestData.inputInterruptId         = TST_DESFLOW_Acceptance_TestData[TST_index].inputInterruptId;
    TST_TestData.outputInterruptId        = TST_DESFLOW_Acceptance_TestData[TST_index].outputInterruptId;
    TST_TestData.inputBigEndianOrderFlag  = TST_DESFLOW_Acceptance_TestData[TST_index].inputBigEndianOrderFlag;
	TST_TestData.inputBuffersArraySize    = TST_DESFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
	TST_TestData.outputBigEndianOrderFlag = TST_DESFLOW_Acceptance_TestData[TST_index].outputBigEndianOrderFlag;
	TST_TestData.outputBuffersArraySize   = TST_DESFLOW_Acceptance_TestData[TST_index].outputBuffersArraySize;
	TST_TestData.wordsBuffersOrderFlag    = TST_DESFLOW_Acceptance_TestData[TST_index].wordsBuffersOrderFlag;
	

	/* initialize flow init parameters */
	CRYS_ATP_TST_FLOW_InitParams(TST_index, DX_TRUE, &TST_TestData, pDESFlowParamsInfo, CRYS_FLOW_DES);
    /* initialize DES engine data */
	CRYS_ATP_TST_FLOW_InitDESEngineData(TST_index);
	
	TST_FLOWDESEngineData.EncryptDecryptFlag = encryptMode;
	
	/* Init DES flow */
	/*****************/
	TST_ATP_Error = CRYS_FLOW_DESInit(flowType,
                        			  pDESFlowParamsInfo,
                        			  &TST_FLOWDESEngineData,  
                        			  TST_DESFlowId);

	ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_DESInit is done\n"));
	
    if (TST_FLOWDESEngineData.EncryptDecryptFlag == CRYS_DES_Encrypt)
        TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_DESInit encrypt",TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
    else
        TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_DESInit decrypt",TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  

	
	return CRYS_OK;     
}

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FinishDESFlow
 *  
 * Inputs:
 *  TST_index          - test index
 *  TST_DESFlowId      - flow id
 *  TST_MasterTestName - test name
 *  blockSize          - block size for finish
 *  pDESFlowParamsInfo - flow params info
 *  encryptMode        = encrypt/decrypt mode
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *	This function finishes DES flow.
 *
 * Algorithm:     
 *	1. Finish DES flow
 *  2. Get results
 *  3. Compare output buffers with expected results
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FinishDESFlow ( int                    TST_index, 
                                         DxUint32_t             TST_DESFlowId, 
                                         char                   *TST_MasterTestName,
                                         DxUint32_t             blockSize,
                                         CRYS_FLOW_Params_t     *pDESFlowParamsInfo,
                                         CRYS_DES_EncryptMode_t encryptMode ) 
{
    DxUint8_t    *pExpectedOut;
    CRYSError_t  TST_TestStatus = TESTPASS;
    int          j = 0; 
    
    /* Finish DES flow with size of the last input buffer */
    /******************************************************/
    TST_ATP_Error = CRYS_FLOW_Finish(TST_DESFlowId,
                                     blockSize);  
                     
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Finish is done\n"));                             
    
    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Finish",TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

    #if CRYS_ATP_TST_EXIT_ON_ERROR
        if (TST_TestStatus != TESTPASS) 
            return TESTFAIL;
    #endif  
                     
           
    /* Get results */
    /***************/
    do
    {
        
        TST_ATP_Error = CRYS_FLOW_GetResults(TST_DESFlowId, 	
                                             DX_NULL,
								             0);
    }while(TST_ATP_Error == DX_SEP_FLOW_GET_RESULT_NOT_READY_ERROR);
    
   
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_GetResults is done\n"));
    
    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_GetResults",TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

    #if CRYS_ATP_TST_EXIT_ON_ERROR
    if (TST_TestStatus != TESTPASS) 
        return TESTFAIL;
    #endif  

    if (encryptMode == CRYS_DES_Encrypt)
		pExpectedOut = TST_DESFLOW_Acceptance_TestData[TST_index].TST_Expected_output_data;
	else
		pExpectedOut = (DxUint8_t*)TST_Input_Data;

            
	/* Compare output buffers with expected results */
	/************************************************/
	for (j = 0; j < pDESFlowParamsInfo->outputBuffersArraySize; ++j)
	{
		/* compare and prints results of test */
		TST_ATP_Error += (CRYSError_t)DX_VOS_MemCmp(
												pDESFlowParamsInfo->outputBuffersArrays[j].buffer_ptr,
                                 				pExpectedOut + 	j*pDESFlowParamsInfo->outputBuffersArrays[j].bufferSize,
                                 				pDESFlowParamsInfo->outputBuffersArrays[j].bufferSize);
                                    
                                     			
		/*Print the outputted and then the expected result after mem compare */
		ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
								   "\nData outputted by engine [output buffer]-- \n",
								   (DxUint8_t*)pDESFlowParamsInfo->outputBuffersArrays[j].buffer_ptr, 
								   pDESFlowParamsInfo->outputBuffersArrays[j].bufferSize); 

 		ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
								  "\nData Expected by engine --\n",
								   (DxUint8_t*)pExpectedOut + 	j*pDESFlowParamsInfo->outputBuffersArrays[j].bufferSize,
								   pDESFlowParamsInfo->outputBuffersArrays[j].bufferSize);                 	                                     			
	}
    
    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

    #if CRYS_ATP_TST_EXIT_ON_ERROR
        if (TST_TestStatus != TESTPASS) 
            return TESTFAIL;
    #endif  
    
    return TST_TestStatus;
}

/*****************************************************************************
* Function Name:                                                           
*  CRYS_ATP_TST_FLOW_CheckDESOperation
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function checks some DES operation
* Algorithm:     
* 1. Execute some DES operation 
* 2. Compare result to expected one
******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_CheckDESOperation () 
{

	/********* TST Variables Definitions ************/
	CRYSError_t   TST_TestStatus = TESTPASS;
	/* use test index = 0 */
	DxUint32_t   TST_index = 0;
	
	/* Run CRYS_DES function */
	/*************************/
	CRYS_DES( TST_DESFLOW_Acceptance_TestData[0].TST_IvCounter,
			  &TST_DESFLOW_Acceptance_TestData[0].TST_Key,
			  TST_DESFLOW_Acceptance_TestData[0].TST_NumOfKeys,	
			  CRYS_DES_Encrypt,
			  TST_DESFLOW_Acceptance_TestData[0].TST_OperationMode,
			  (DxUint8_t*)TST_Input_Data,
			  TST_FLOW_InputDataSize,
		      (DxUint8_t*)TST_FLOW_OutputDataForIntegratedEngines);

	/* compare DES result with one expected */
	/****************************************/
	TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(
		TST_FLOW_OutputDataForIntegratedEngines,
		TST_DESFLOW_Acceptance_TestData[0].TST_Expected_output_data,
		TST_FLOW_InputDataSize);					      

	ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CheckDESOperation is done\n"));
    
    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CheckDESOperation",TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name);

    #if CRYS_ATP_TST_EXIT_ON_ERROR
        if (TST_TestStatus != TESTPASS) 
            return TESTFAIL;
    #endif  
	return TST_ATP_Error;

}


/************************ Public Functions ******************************/

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_DES_ONETEST_Acceptance
 *  
 * Inputs:
 *  TST_index          - test index
 *  TST_MasterTestName - test name
 *  flowType           - flow type (static or dynamic)
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for static DES FLOW API's
 
 * Algorithm:     
 *  1. Init DES flow
 *  2. Send interrupts that input and output buffers are ready
 *  3. Finish DES flow, read data from the output buffers and compare to expected results
 *  4. Terminate flow
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_DES_ONETEST_Acceptance ( int              TST_index, 
                                                       char             *TST_MasterTestName,
                                                       CRYS_FLOW_Type_t flowType) 
{
    /********* TST Variables Definitions ************/
	CRYSError_t             	TST_TestStatus = TESTPASS;
	
	/* flow id */
	DxUint32_t                   flowId = 0;
	DxUint32_t                   i = 0;
	DxUint16_t                   blockSize;
	

	/************************************************/
	/*********** Start of test **********************/
	/************************************************/
	for (TST_ATP_EncryptDecryptFlag = 0; TST_ATP_EncryptDecryptFlag < 2 ; TST_ATP_EncryptDecryptFlag++)     /*Select mode or operation  
																											TST_OperationMode = 0 == Encrypt
																											TST_OperationMode = 1 == Decrypt*/
	{
	    /* input block size */
        blockSize       = TST_FLOW_InputDataSize/TST_DESFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
		

		/* initialize input buffer: */
		if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
			DX_VOS_FastMemCpy((DxUint8_t*)TST_FLOW_InputData, TST_Input_Data, TST_FLOW_InputDataSize);
		else
			DX_VOS_FastMemCpy((DxUint8_t*)TST_FLOW_InputData, TST_DESFLOW_Acceptance_TestData[TST_index].TST_Expected_output_data, TST_FLOW_InputDataSize);
		
		/* Initialize DES flow parameters and init DES flow by calling function */
		/* CRYS_FLOW_DESInit                                                    */
		/************************************************************************/
		TST_TestStatus = CRYS_ATP_TST_InitDESFlow(TST_index, flowType, TST_MasterTestName, &flowId, &TST_FLOWParamsInfo, (CRYS_DES_EncryptMode_t)TST_ATP_EncryptDecryptFlag);
		
#if CRYS_ATP_TST_EXIT_ON_ERROR
    	if (TST_TestStatus != TESTPASS) 
    		return TESTFAIL;
#endif  

		
		/* Send interrupts that input and output buffers are ready */
		/***********************************************************/
		

		/* Send interrupt that input buffers are ready */
		if (TST_FLOW_IsFinishSizeZero == DX_TRUE)
		{
		    for (i = 0; i < TST_FLOWParamsInfo.inputBuffersArraySize; ++i)
		        CRYS_ATP_TST_FLOW_SendInterrupt(TST_FLOWParamsInfo.inputInterruptId);
		        
            blockSize = 0;		        
		}
		else
		{
		    for (i = 0; i < TST_FLOWParamsInfo.inputBuffersArraySize - 1; ++i)
		        CRYS_ATP_TST_FLOW_SendInterrupt(TST_FLOWParamsInfo.inputInterruptId);
		}


		/* Send interrupt that output buffers are ready */
		for (i = 0; i < TST_FLOWParamsInfo.outputBuffersArraySize; ++i)
		    CRYS_ATP_TST_FLOW_SendInterrupt(TST_FLOWParamsInfo.outputInterruptId);

			

		/* Finish DES flow, get results and compare results with expected  */
        /* by calling functions CRYS_FLOW_Finish and CRYS_FLOW_GetResults  */
        /*******************************************************************/
		TST_TestStatus = CRYS_ATP_TST_FinishDESFlow(TST_index, flowId, TST_MasterTestName, blockSize, &TST_FLOWParamsInfo, (CRYS_DES_EncryptMode_t)TST_ATP_EncryptDecryptFlag);
		
#if CRYS_ATP_TST_EXIT_ON_ERROR
    	if (TST_TestStatus != TESTPASS) 
    		return TESTFAIL;
#endif 		
		

        /* Terminate flow */
        /******************/
		TST_ATP_Error = CRYS_FLOW_Terminate(flowId);
		
		ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Terminate is done\n"));                             
		
		TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Terminate",TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
#endif  

	} /* End of TST_ATP_EncryptDecryptFlag*/ 

	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_DES_ONETEST_Acceptance*/ 

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_DESSTATIC_Acceptance
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for static DES FLOW API's
 
 * Algorithm:     
 *	This function goes through all tests vectors and execute DES static flow acceptance test
 *  for each vector
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_DESSTATIC_Acceptance ( void ) 
{

	/********* TST Variables Definitions ************/
	int                     	TST_index;
	CRYSError_t             	TST_TestStatus = TESTPASS;
	char                     	TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_DESSTATIC_Acceptance"; 
	
	
	/* initialize input and output buffers */
	TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;
	TST_FLOWDESEngineData.Key_ptr          = DX_NULL;


	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n"));
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_DESSTATIC_Acceptance Tests ********************* \n"));      
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n \n"));


	/* go through all tests vectors and execute DES flow acceptance test for each vector */
    /*************************************************************************************/
	for (TST_index =0 ; TST_index < TST_NumberOf_DESFLOW_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
	{
		TST_TestStatus = CRYS_ATP_TST_FLOW_DES_ONETEST_Acceptance(TST_index, TST_MasterTestName, CRYS_FLOW_TypeStatic);
		

#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
#endif  

	}/* End of TST_index */     

	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_DESSTATIC_Acceptance*/ 



/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_DESDYNAMIC_Acceptance
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for static DES FLOW API's
 
 * Algorithm:     
 *	This function goes through all tests vectors and execute DES dynamic flow acceptance test
 *  for each vector
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_DESDYNAMIC_Acceptance ( void ) 
{

	/********* TST Variables Definitions ************/
	int                     	TST_index;
	CRYSError_t             	TST_TestStatus = TESTPASS;
	char                     	TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_DESDYNAMIC_Acceptance"; 
	
	
	/* initialize input and output buffers */
	TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;
	TST_FLOWDESEngineData.Key_ptr          = DX_NULL;


	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n"));
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_DESDYNAMIC_Acceptance Tests ********************* \n"));      
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n \n"));


	/* go through all tests vectors and execute AES flow acceptance test for each vector */
    /*************************************************************************************/
	for (TST_index =0 ; TST_index < TST_NumberOf_DESFLOW_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
	{
		TST_TestStatus = CRYS_ATP_TST_FLOW_DES_ONETEST_Acceptance(TST_index, TST_MasterTestName, CRYS_FLOW_TypeDynamic);
		

#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
#endif  

	}/* End of TST_index */     

	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_DESDYNAMIC_Acceptance*/ 


/*****************************************************************************
* Function Name:                                                           
*  CRYS_ATP_TST_FLOW_DESDYNAMIC_AddBuffers_Acceptance
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for dynamic AES FLOW API's

* Algorithm:     
*  1. Initialize DES flow with dynamic type and other parameters
*  2. Init DES flow
*  3. Send interrupts that input and output buffers are ready
*  4. Add buffers to dynamic flow
*  5. Send interrupts that added input and output buffers are ready
*  6. Finish DES flow
*  7. Get results
*  8. Read data from the output buffers and compare to expected results
******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_DESDYNAMIC_AddBuffers_Acceptance ( void ) 
{

	/********* TST Variables Definitions ************/
	int                     	 TST_index;
	CRYSError_t             	 TST_TestStatus = TESTPASS;
	char                     	 TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_DESDYNAMIC_AddBuffers_Acceptance"; 
	DxUint8_t                   *pExpectedOut;
	/* flow id */
	DxUint32_t                   flowId = 0;
	DxUint32_t                   j = 0; 
	DxUint32_t                   inputBuffersArraySize = 1;
	DxUint32_t                   outputBuffersArraySize = 1;
    CRYS_FLOW_BufferData_t       inputBuffersArray_ptr[TST_FLOW_DATA_MAX_NUM_OF_FLOWBUFFERS];
    CRYS_FLOW_BufferData_t       outputBuffersArray_ptr[TST_FLOW_DATA_MAX_NUM_OF_FLOWBUFFERS];
    DxUint16_t                   blockSize;
	DxUint16_t                   outputBlockSize;

	/* initialize input and output buffers */
	TST_FLOWParamsInfo.inputBuffersArrays = DX_NULL;
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;

	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n"));
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_DESDYNAMIC_AddBuffers_Acceptance Tests ********************* \n"));      
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n \n"));


	/************************************************/
	/*********** Start of test **********************/
	/************************************************/
	for (TST_index =0 ; TST_index < TST_NumberOf_DESFLOW_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
	{
		TST_TestData.inputBigEndianOrderFlag  = TST_DESFLOW_Acceptance_TestData[TST_index].inputBigEndianOrderFlag;
		TST_TestData.inputBuffersArraySize    = TST_DESFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
		TST_TestData.inputInterruptId         = TST_DESFLOW_Acceptance_TestData[TST_index].inputInterruptId;
		TST_TestData.outputBigEndianOrderFlag = TST_DESFLOW_Acceptance_TestData[TST_index].outputBigEndianOrderFlag;
		TST_TestData.outputBuffersArraySize   = TST_DESFLOW_Acceptance_TestData[TST_index].outputBuffersArraySize;
		TST_TestData.outputInterruptId        = TST_DESFLOW_Acceptance_TestData[TST_index].outputInterruptId;
		TST_TestData.wordsBuffersOrderFlag    = TST_DESFLOW_Acceptance_TestData[TST_index].wordsBuffersOrderFlag;

		for (TST_ATP_EncryptDecryptFlag = 0; TST_ATP_EncryptDecryptFlag < 2 ; TST_ATP_EncryptDecryptFlag++)     /*Select mode or operation  
																												TST_OperationMode = 0 == Encrypt
																												TST_OperationMode = 1 == Decrypt*/
		{
			/* Initialize DES flow with dynamic type and other parameters */
         	/**************************************************************/
            blockSize       = TST_FLOW_InputDataSize/TST_DESFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
			outputBlockSize = TST_FLOW_InputDataSize/TST_DESFLOW_Acceptance_TestData[TST_index].outputBuffersArraySize;

			/* prepare input buffer: */
			if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
				DX_VOS_FastMemCpy((DxUint8_t*)TST_FLOW_InputData, TST_Input_Data, TST_FLOW_InputDataSize);
			else
				DX_VOS_FastMemCpy((DxUint8_t*)TST_FLOW_InputData, TST_DESFLOW_Acceptance_TestData[TST_index].TST_Expected_output_data, TST_FLOW_InputDataSize);

			/* initialize flow init parameters */
			CRYS_ATP_TST_FLOW_InitParams(TST_index, DX_FALSE, &TST_TestData, &TST_FLOWParamsInfo, CRYS_FLOW_DES);

			/* initialize DES engine data */
			CRYS_ATP_TST_FLOW_InitDESEngineData(TST_index);

			TST_FLOWDESEngineData.EncryptDecryptFlag = (CRYS_DES_EncryptMode_t)TST_ATP_EncryptDecryptFlag;

			/* Init DES flow */
			/*****************/
			TST_ATP_Error = CRYS_FLOW_DESInit(CRYS_FLOW_TypeDynamic,
				&TST_FLOWParamsInfo,
				&TST_FLOWDESEngineData,  
				&flowId);


			ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_InitDES is done\n"));

			TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_InitDES",TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
			if (TST_TestStatus != TESTPASS) 
				return TESTFAIL;
#endif  

			/* Send interrupts that input and output buffers are ready */
            /***********************************************************/
            
			/* Send interrupt that input buffers are ready */
            CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);
            
            
            /* Send interrupt that output buffers are ready */
            CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.outputInterruptId);


			inputBuffersArraySize  = TST_DESFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
			outputBuffersArraySize = TST_DESFLOW_Acceptance_TestData[TST_index].outputBuffersArraySize;

			/* prepare buffers to add */
			for (j = 0; j < inputBuffersArraySize - 1; ++j )
			{
				/* point to the second block of the input buffer */
				inputBuffersArray_ptr[j].buffer_ptr = (DxUint8_t*)TST_FLOW_InputData + (j+1)*blockSize;
	                
				inputBuffersArray_ptr[j].bufferSize = blockSize;                
			}

		
			for (j = 0; j < outputBuffersArraySize - 1; ++j )
			{
				/* point to the second block of the output data */
				outputBuffersArray_ptr[j].buffer_ptr = (DxUint8_t*)TST_FLOW_OutputData[CRYS_FLOW_DES] + (j+1)*outputBlockSize;

				outputBuffersArray_ptr[j].bufferSize = outputBlockSize;                
			}

		
			/* Add buffers to dynamic flow */
            /*******************************/
			TST_ATP_Error = CRYS_FLOW_AddBuffers(flowId,
                            				inputBuffersArray_ptr,
                            				inputBuffersArraySize - 1,
                            				outputBuffersArray_ptr,
                            				outputBuffersArraySize - 1);
                            				
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_AddBuffers is done\n"));

			TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_AddBuffers",TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
			if (TST_TestStatus != TESTPASS) 
				return TESTFAIL;
#endif  
                            				
                            				
			/* Send interrupts that added input and output buffers are ready */
            /*****************************************************************/
            	
            /* Send interrupt that input buffers are ready */
            for (j = 0; j < inputBuffersArraySize - 1; ++j )
                CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);
                    
            blockSize = 0;    
            
            /* Send interrupt that output buffers are ready */
            for (j = 0; j < outputBuffersArraySize - 1; ++j )
                CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.outputInterruptId);                                 

			/* Finish DES flow */
			/*******************/
			TST_ATP_Error = CRYS_FLOW_Finish(flowId,
				                             blockSize); 

			ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Finish is done\n"));                             
			
			TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Finish",TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
			if (TST_TestStatus != TESTPASS) 
				return TESTFAIL;
#endif  


			/* Get results */
			/***************/
			do
            {
                
                TST_ATP_Error = CRYS_FLOW_GetResults(flowId,
				                                 (DxUint8_t*)TST_FLOW_Hash_OutputData,
				                                 CRYS_HASH_RESULT_SIZE_IN_WORDS);
            }while(TST_ATP_Error == DX_SEP_FLOW_GET_RESULT_NOT_READY_ERROR);
			
			
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_GetResults is done\n"));
            
            TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_GetResults",TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
			if (TST_TestStatus != TESTPASS) 
				return TESTFAIL;
#endif  
			
			/* terminate flow */
			TST_ATP_Error = CRYS_FLOW_Terminate(flowId);
			
		    ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Terminate is done\n"));                             
			
			TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Terminate",TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
			if (TST_TestStatus != TESTPASS) 
				return TESTFAIL;
#endif  	

			/* Read data from the output buffers and compare to expected result */
			/********************************************************************/
			if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
				pExpectedOut = TST_DESFLOW_Acceptance_TestData[TST_index].TST_Expected_output_data;
			else
				pExpectedOut = (DxUint8_t*)TST_Input_Data;

			for (j = 0; j < 1; ++j)
			{
				/* compare and prints results of test */
				TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(
					TST_FLOWParamsInfo.outputBuffersArrays[j].buffer_ptr,
					pExpectedOut + 	j*TST_FLOWParamsInfo.outputBuffersArrays[j].bufferSize,
					TST_FLOWParamsInfo.outputBuffersArrays[j].bufferSize);


				/*Print the outputted and then the expected result after mem compare */
				ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
					"\nData outputted by engine [output buffer]-- \n",
					(DxUint8_t*)TST_FLOWParamsInfo.outputBuffersArrays[j].buffer_ptr, 
					TST_FLOWParamsInfo.outputBuffersArrays[j].bufferSize); 

				ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
					"\nData Expected by engine --\n",
					(DxUint8_t*)pExpectedOut + 	j*TST_FLOWParamsInfo.outputBuffersArrays[j].bufferSize,
					TST_FLOWParamsInfo.outputBuffersArrays[j].bufferSize);                 	                                     			

				
			}

			/* compare and prints results of test */
			for (j = 0; j < inputBuffersArraySize - 1; ++j )
			{
				TST_ATP_Error += (CRYSError_t)DX_VOS_MemCmp(
					outputBuffersArray_ptr[j].buffer_ptr,
					pExpectedOut + 	(j+1)*outputBuffersArray_ptr[j].bufferSize,
					outputBuffersArray_ptr[j].bufferSize);
			}
			
			TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
			if (TST_TestStatus != TESTPASS) 
				return TESTFAIL;
#endif  

		} /* End of TST_ATP_EncryptDecryptFlag*/ 

	}/* End of TST_index */     

	
	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_DESDYNAMIC_AddBuffers_Acceptance*/

/*****************************************************************************
* Function Name:                                                           
*  CRYS_ATP_TST_FLOW_DES_ONETEST_Acceptance_WithAdditionalEngines
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for DES FLOW API's with additional engines.

* Algorithm:     
*  1. Init DES flow with type and other parameters
*  2. Send several interrupts that input and output buffers are ready
*  3. Execute some HASH operation and compare the result of the HASH to expected
*  4. Send several interrupts that input and output buffers are ready, 
*  5. Execute some AES operation and compare the result of the DES to expected
*  6. Send several interrupts that input and output buffers are ready, 
*  7. Execute some DES operation and compare the result of the AES to expected
*  8. Finish DES flow 
*  9. Execute some AES operation and compare with expected result
*  10. Get result
*  11.Terminate flow
*  12.Read data from the output buffers and compare to expected results
******************************************************************************/
CRYSError_t CRYS_ATP_TST_FLOW_DES_ONETEST_Acceptance_WithAdditionalEngines ( 
                                                                int              TST_index, 
                                                                char             *TST_MasterTestName,
                                                                CRYS_FLOW_Type_t flowType ) 
{

	/********* TST Variables Definitions ************/
    CRYSError_t             	 TST_TestStatus = TESTPASS;
	DxUint8_t                   *pExpectedOut;
	/* flow id */
	DxUint32_t                   flowId = 0;
    int                          j = 0; 
    int                          i = 0;
    DxUint16_t                   blockSize;
	

	/************************************************/
	/*********** Start of test **********************/
	/************************************************/
	TST_TestData.inputBigEndianOrderFlag  = TST_DESFLOW_Acceptance_TestData[TST_index].inputBigEndianOrderFlag;
	TST_TestData.inputBuffersArraySize    = TST_DESFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
	TST_TestData.inputInterruptId         = TST_DESFLOW_Acceptance_TestData[TST_index].inputInterruptId;
	TST_TestData.outputBigEndianOrderFlag = TST_DESFLOW_Acceptance_TestData[TST_index].outputBigEndianOrderFlag;
	TST_TestData.outputBuffersArraySize   = TST_DESFLOW_Acceptance_TestData[TST_index].outputBuffersArraySize;
	TST_TestData.outputInterruptId        = TST_DESFLOW_Acceptance_TestData[TST_index].outputInterruptId;
	TST_TestData.wordsBuffersOrderFlag    = TST_DESFLOW_Acceptance_TestData[TST_index].wordsBuffersOrderFlag;

	for (TST_ATP_EncryptDecryptFlag = 0; TST_ATP_EncryptDecryptFlag < 2 ; TST_ATP_EncryptDecryptFlag++)     /*Select mode or operation  
																											TST_OperationMode = 0 == Encrypt
																											TST_OperationMode = 1 == Decrypt*/
	{
		/* input block size */
		blockSize       = TST_FLOW_InputDataSize/TST_DESFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
		

		/* prepare input buffer: */
		if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
			DX_VOS_FastMemCpy((DxUint8_t*)TST_FLOW_InputData, TST_Input_Data, TST_FLOW_InputDataSize);
		else
		{
			DX_VOS_FastMemCpy((DxUint8_t*)TST_FLOW_InputData, TST_DESFLOW_Acceptance_TestData[TST_index].TST_Expected_output_data, TST_FLOW_InputDataSize);
		}
		
		/* Initialize DES flow parameters and init AES flow by calling function */
		/* CRYS_FLOW_DESInit                                                    */
		/************************************************************************/
		TST_TestStatus = CRYS_ATP_TST_InitDESFlow(TST_index, flowType, TST_MasterTestName, &flowId, &TST_FLOWParamsInfo, (CRYS_DES_EncryptMode_t)TST_ATP_EncryptDecryptFlag);
		
		#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
        #endif  

	    /* Send interrupts that input and output buffers are ready */
	    /***********************************************************/
	    
	    /* Send interrupt that input buffers are ready */
	    CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);


		/* Send interrupt that output buffers are ready */
		CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.outputInterruptId);

		
		/* execute HASH operation and compare with expected result */
		/***********************************************************/
	    TST_ATP_Error = CRYS_ATP_TST_FLOW_CheckHASHOperation();
	    
	    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"HASH operation",TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  

        /* Send interrupts that input and output buffers are ready */
	    /***********************************************************/

        /* Send interrupt that input buffers are ready */
	    CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);


		/* Send interrupt that output buffers are ready */
		CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.outputInterruptId);
		
		
		/* execute AES operation and compare with expected result */
		/***********************************************************/
		TST_ATP_Error = CRYS_ATP_TST_FLOW_CheckAESOperation();
	
	    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"AES operation",TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	    if (TST_TestStatus != TESTPASS) 
		    return TESTFAIL;
#endif  
        
        /* Send interrupts that input and output buffers are ready */
	    /***********************************************************/
	    
        /* Send interrupt that input buffers are ready */
        if ((TST_FLOW_IsFinishSizeZero == DX_TRUE) ||
            (TST_FLOWParamsInfo.inputBuffersArraySize < 3)) 
        {
            for (i = 0; i < TST_FLOWParamsInfo.inputBuffersArraySize - 2; ++i)
    	    {
    	        CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);
    	    }
    	    
    	    blockSize = 0;
        }
        else
        {
            for (i = 0; i < TST_FLOWParamsInfo.inputBuffersArraySize - 3; ++i)
    	    {
    	        CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);
    	    }
        }
	    
	    /* Send interrupt that output buffers are ready */
	    for (i = 0; i < TST_FLOWParamsInfo.outputBuffersArraySize - 2; ++i)
	    {
	        CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.outputInterruptId);
	    }
	    
	    
	    /* execute DES operation and compare with expected result */
		/***********************************************************/
	    TST_ATP_Error = CRYS_ATP_TST_FLOW_CheckDESOperation();
	
	    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"DES operation",TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	    if (TST_TestStatus != TESTPASS) 
		    return TESTFAIL;
#endif  

		/* Finish DES flow */
		/*******************/
		TST_ATP_Error = CRYS_FLOW_Finish(flowId,
			                             blockSize);
			                               
		ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Finish is done\n"));                             
	
	    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Finish",TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  	                             
        				                             
		ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Finish is done\n"));                             

		/* execute some AES operation and compare with expected result*/
		/**************************************************************/
		TST_ATP_Error = CRYS_ATP_TST_FLOW_CheckAESOperation();

		TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"AES Operation",TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
#endif  

		
		/* Get results */
		/***************/
		do
        {
            
            TST_ATP_Error = CRYS_FLOW_GetResults(flowId,
			                                     (DxUint8_t*)TST_FLOW_Hash_OutputData,
			                                     CRYS_HASH_RESULT_SIZE_IN_WORDS);
        }while(TST_ATP_Error == DX_SEP_FLOW_GET_RESULT_NOT_READY_ERROR);
		
		
		ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_GetResults is done\n"));
		
		TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_GetResults",TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  
		
		/* Terminate flow */
		/******************/
		TST_ATP_Error = CRYS_FLOW_Terminate(flowId);
	
	    ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Terminate is done\n"));                             
	
	    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Terminate",TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	    if (TST_TestStatus != TESTPASS) 
		    return TESTFAIL;
#endif  

		/* Read data from the output buffers and compare to expected result */
		/********************************************************************/
		if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
			pExpectedOut = TST_DESFLOW_Acceptance_TestData[TST_index].TST_Expected_output_data;
		else
			pExpectedOut = (DxUint8_t*)TST_Input_Data;

		for (j = 0; j < TST_FLOWParamsInfo.outputBuffersArraySize; ++j)
		{
			/* compare and prints results of test */
			TST_ATP_Error += (CRYSError_t)DX_VOS_MemCmp(
				TST_FLOWParamsInfo.outputBuffersArrays[j].buffer_ptr,
				pExpectedOut + 	j*TST_FLOWParamsInfo.outputBuffersArrays[j].bufferSize,
				TST_FLOWParamsInfo.outputBuffersArrays[j].bufferSize);


			/*Print the outputted and then the expected result after mem compare */
			ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
				"\nData outputted by engine [output buffer]-- \n",
				(DxUint8_t*)TST_FLOWParamsInfo.outputBuffersArrays[j].buffer_ptr, 
				TST_FLOWParamsInfo.outputBuffersArrays[j].bufferSize); 

			ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
				"\nData Expected by engine --\n",
				(DxUint8_t*)pExpectedOut + 	j*TST_FLOWParamsInfo.outputBuffersArrays[j].bufferSize,
				TST_FLOWParamsInfo.outputBuffersArrays[j].bufferSize);                 	                                     			
		}
		
		TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
#endif  

	} /* End of TST_ATP_EncryptDecryptFlag*/ 

	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_DES_ONETEST_Acceptance_WithAdditionalEngines*/ 

/*****************************************************************************
* Function Name:                                                           
*  CRYS_ATP_TST_FLOW_DESSTATIC_Acceptance_WithAdditionalEngines
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for static DES FLOW API's with additional engines.

* Algorithm:     
*  This function goes through all flow vectors and execute acceptance test with 
*  additional engines for each vector
******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_DESSTATIC_Acceptance_WithAdditionalEngines ( void ) 
{

	/********* TST Variables Definitions ************/
	int                     	 TST_index;
	CRYSError_t             	 TST_TestStatus = TESTPASS;
	char                     	 TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_DESSTATIC_Acceptance_WithAdditionalEngines"; 


	/* initialize input and output buffers */
	TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;


	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"******************************************************************************************* \n"));
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_DESSTATIC_Acceptance_WithAdditionalEngines Tests ********************* \n"));      
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"******************************************************************************************* \n \n"));


	/* go through all flow vectors and execute acceptance test */
	for (TST_index =0 ; TST_index < TST_NumberOf_DESFLOW_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
	{
		TST_TestStatus = CRYS_ATP_TST_FLOW_DES_ONETEST_Acceptance_WithAdditionalEngines(
		                                                           TST_index,
	                                                               TST_MasterTestName,
	                                                               CRYS_FLOW_TypeStatic);
	                                                               
#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
#endif  
	                                                               
	}/* End of TST_index */     

	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_DESSTATIC_Acceptance_WithAdditionalEngines*/ 


/*****************************************************************************
* Function Name:                                                           
*  CRYS_ATP_TST_FLOW_DESDYNAMIC_Acceptance_WithAdditionalEngines
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for static DES FLOW API's with additional engines.

* Algorithm:     
*  This function goes through all flow vectors and execute acceptance test with 
*  additional engines for each vector
******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_DESDYNAMIC_Acceptance_WithAdditionalEngines ( void ) 
{

	/********* TST Variables Definitions ************/
	int                     	 TST_index;
	CRYSError_t             	 TST_TestStatus = TESTPASS;
	char                     	 TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_DESDYNAMIC_Acceptance_WithAdditionalEngines"; 
	
	/* initialize input and output buffers */
	TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;


	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"******************************************************************************************* \n"));
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_DESDYNAMIC_Acceptance_WithAdditionalEngines Tests ********************* \n"));      
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"******************************************************************************************* \n \n"));


	/* go through all flow vectors and execute acceptance test */
	for (TST_index =0 ; TST_index < TST_NumberOf_DESFLOW_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
	{
		TST_TestStatus = CRYS_ATP_TST_FLOW_DES_ONETEST_Acceptance_WithAdditionalEngines(
		                                                           TST_index,
	                                                               TST_MasterTestName,
	                                                               CRYS_FLOW_TypeDynamic);
	                                                               
#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
#endif  
	                                                               
	}/* End of TST_index */     

	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_DESDYNAMIC_Acceptance_WithAdditionalEngines*/ 


/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_DESSTATIC_ONETEST_Cyclic_Acceptance
 *  
 * Inputs:
 *  TST_index          - test index
 *  TST_MasterTestName - test name 
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for static cyclic DES FLOW API's
 
 * Algorithm:     
 * 1. Initialize DES flow with static type and other parameters 
 * 2. Init DES flow
 * 3. Send interrupts to execute cyclic flow operation 
 * 4. Finish DES flow
 * 5. Get results
 * 6. Terminate flow
 * 7. Read data from the output buffers and compare to expected result
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_DESSTATIC_ONETEST_Cyclic_Acceptance ( int   TST_index,
                                                                    char *TST_MasterTestName  ) 
{

	/********* TST Variables Definitions ************/
    CRYSError_t             	 TST_TestStatus = TESTPASS;
	DxUint8_t                   *pExpectedOut;
	/* flow id */
	DxUint32_t                   flowId = 0;
	DxUint32_t                   blockSize = 0;
	DxUint32_t                   j = 0; 
	DxUint32_t                   i = 0; 
	
	volatile DxUint32_t          delay = 0;
	
	/* output to accumulate result from cyclic operation */
	DxUint32_t	                 TST_FLOW_Cyclic_OutputData[TST_FLOW_DATA_VEC_SIZE_IN_BYTS/4];
	DxUint8_t                    *pAccumulatorOutputData = (DxUint8_t*)TST_FLOW_Cyclic_OutputData;
    
    
    /* initialize input and output buffers */
	TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;
	
	
	TST_FLOWParamsInfo.inputBigEndianOrderFlag  = TST_DESFLOW_Acceptance_TestData[TST_index].inputBigEndianOrderFlag;
	TST_FLOWParamsInfo.inputBuffersArraySize    = TST_DESFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
	TST_FLOWParamsInfo.inputInterruptId         = TST_DESFLOW_Acceptance_TestData[TST_index].inputInterruptId;
	TST_FLOWParamsInfo.outputBigEndianOrderFlag = TST_DESFLOW_Acceptance_TestData[TST_index].outputBigEndianOrderFlag;
	TST_FLOWParamsInfo.outputBuffersArraySize   = TST_DESFLOW_Acceptance_TestData[TST_index].outputBuffersArraySize;
	TST_FLOWParamsInfo.outputInterruptId        = TST_DESFLOW_Acceptance_TestData[TST_index].outputInterruptId;
	TST_FLOWParamsInfo.wordsBuffersOrderFlag    = TST_DESFLOW_Acceptance_TestData[TST_index].wordsBuffersOrderFlag;

	for (TST_ATP_EncryptDecryptFlag = 0; TST_ATP_EncryptDecryptFlag < 1; TST_ATP_EncryptDecryptFlag++)     /*Select mode or operation  
                                                                                                              TST_OperationMode = 0 == Encrypt
                                                                                                              TST_OperationMode = 1 == Decrypt*/
    {
        /* pointer to cyclic output data for accumulating */
        pAccumulatorOutputData = (DxUint8_t*)TST_FLOW_Cyclic_OutputData;
        
        /* prepare input buffer: */
		if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
			DX_VOS_FastMemCpy((DxUint8_t*)TST_FLOW_InputData, TST_Input_Data, TST_FLOW_InputDataSize);
		else
		{
			/* in modes below there is only encrypt option */
			if ((TST_FLOWDESEngineData.OperationMode == CRYS_AES_MAC_mode)      ||
				(TST_FLOWDESEngineData.OperationMode == CRYS_AES_CTR_mode)      ||
				(TST_FLOWDESEngineData.OperationMode == CRYS_AES_XCBC_MAC_mode) ||
				(TST_FLOWDESEngineData.OperationMode == CRYS_AES_CMAC_mode))
				break;

			DX_VOS_FastMemCpy((DxUint8_t*)TST_FLOW_InputData, TST_DESFLOW_Acceptance_TestData[TST_index].TST_Expected_output_data, TST_FLOW_InputDataSize);
		}
        
     	/* Initialize DES flow data */
     	/****************************/
     	
     	/* initialize input and output buffers */
     	blockSize = TST_FLOW_DATA_VEC_SIZE_IN_BYTS/4;
     	
     	for (j = 0; j < 2; ++j )
		{
			/* point to the second block of the input buffer */
			TST_InputBuffers[j].buffer_ptr = (DxUint8_t*)TST_FLOW_InputData + (j)*blockSize;
                
			TST_InputBuffers[j].bufferSize = blockSize;                
		}
        
       
        for (j = 0; j < 2; ++j )
		{
			/* point to the second block of the output data */
			TST_OutputBuffers[CRYS_FLOW_DES][j].buffer_ptr = (DxUint8_t*)TST_FLOW_OutputData[CRYS_FLOW_DES] + (j)*blockSize;
        
			TST_OutputBuffers[CRYS_FLOW_DES][j].bufferSize = blockSize;                
		}
     	
        TST_FLOWParamsInfo.inputBuffersArrays    = TST_InputBuffers;
	    TST_FLOWParamsInfo.inputBuffersArraySize = 2;
	    
	    TST_FLOWParamsInfo.outputBuffersArrays   = TST_OutputBuffers[CRYS_FLOW_DES];
	    TST_FLOWParamsInfo.outputBuffersArraySize = 2;

	        	    	
     	/* initialize DES engine data */
     	CRYS_ATP_TST_FLOW_InitDESEngineData(TST_index);
     	
     
     	TST_FLOWDESEngineData.EncryptDecryptFlag = (CRYS_DES_EncryptMode_t)TST_ATP_EncryptDecryptFlag;
     	
	    /* Init DES flow */
     	/*****************/
     	TST_ATP_Error = CRYS_FLOW_DESInit(CRYS_FLOW_TypeStatic,
                     	                  &TST_FLOWParamsInfo,
                                          &TST_FLOWDESEngineData,  
                                          &flowId);
                          
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_InitDES is done\n"));
                            
        TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_InitDES",TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
 	
        #if CRYS_ATP_TST_EXIT_ON_ERROR
            if (TST_TestStatus != TESTPASS) 
                return TESTFAIL;
        #endif  
        
        /* Send interrupts to execute cyclic flow operation */
        /****************************************************/
        
        /* several times send input and output interupts to execute */
        /* several time flow operation */
        for (j = 0; j < 2; j++)
        {
            /* Send interrupt that input buffers are ready */
            CRYS_ATP_TST_FLOW_SendInterrupt(TST_FLOWParamsInfo.inputInterruptId);
            
            /* Send interrupt that input buffers are ready */
            CRYS_ATP_TST_FLOW_SendInterrupt(TST_FLOWParamsInfo.inputInterruptId);
            
            CRYS_ATP_TST_FLOW_SendInterrupt(TST_FLOWParamsInfo.outputInterruptId);
                
            CRYS_ATP_TST_FLOW_SendInterrupt(TST_FLOWParamsInfo.outputInterruptId);
			    
		
            /* wating for DMA operation to be finished and correct
               output data to be updated  */                       
            for (delay = 0; delay < 100000; delay++);
            
            /* accumulate output data after each cycle iteration */
            for (i = 0; i < TST_FLOWParamsInfo.outputBuffersArraySize; ++i)
			{
			    DX_VOS_FastMemCpy(pAccumulatorOutputData,
                                  TST_FLOWParamsInfo.outputBuffersArrays[i].buffer_ptr,
                                  TST_FLOWParamsInfo.outputBuffersArrays[i].bufferSize);       
                                  
                pAccumulatorOutputData += TST_FLOWParamsInfo.outputBuffersArrays[i].bufferSize;                                                                                
			}
		
        }
		    
        
        /* Finish DES flow */
        /*******************/
        TST_ATP_Error = CRYS_FLOW_Finish(flowId,
                                         0);  
                         
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Finish is done\n"));                             
        
        TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Finish",TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
 	
        #if CRYS_ATP_TST_EXIT_ON_ERROR
            if (TST_TestStatus != TESTPASS) 
                return TESTFAIL;
        #endif  
                         
       
        /* Get results */
        /***************/
        do
        {
            
            TST_ATP_Error = CRYS_FLOW_GetResults(flowId,
			                                 (DxUint8_t*)TST_FLOW_Hash_OutputData,
			                                 CRYS_HASH_RESULT_SIZE_IN_WORDS);
        }while(TST_ATP_Error == DX_SEP_FLOW_GET_RESULT_NOT_READY_ERROR);
        
        
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_GetResults is done\n"));
        
        TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_GetResults",TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
 	
        #if CRYS_ATP_TST_EXIT_ON_ERROR
            if (TST_TestStatus != TESTPASS) 
                return TESTFAIL;
        #endif  
        
        /* Terminate flow */
        /******************/
        TST_ATP_Error = CRYS_FLOW_Terminate(flowId);
        
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Terminate is done\n"));                             
        
        TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Terminate",TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
 	
        #if CRYS_ATP_TST_EXIT_ON_ERROR
            if (TST_TestStatus != TESTPASS) 
                return TESTFAIL;
        #endif  
        
        /* Read data from the output buffers and compare to expected result */
        /********************************************************************/                             
		if (TST_ATP_EncryptDecryptFlag == TST_ENCRYPT)
			pExpectedOut = TST_DESFLOW_Acceptance_TestData[TST_index].TST_Expected_cyclicOperation_data;
		else
			pExpectedOut = (DxUint8_t*)TST_Input_Data;

        /* compare and prints results of test */
    	TST_ATP_Error += (CRYSError_t)DX_VOS_MemCmp(
													TST_FLOW_Cyclic_OutputData,
                                     				pExpectedOut,
                                     				TST_FLOW_DATA_VEC_SIZE_IN_BYTS);
	                                    
	                                     			
		/*Print the outputted and then the expected result after mem compare */
		ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
								   "\nData outputted by engine [output buffer]-- \n",
								   (DxUint8_t*)TST_FLOW_Cyclic_OutputData, 
								   TST_FLOW_DATA_VEC_SIZE_IN_BYTS); 

 		ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
								  "\nData Expected by engine --\n",
								   (DxUint8_t*)pExpectedOut,
								   TST_FLOW_DATA_VEC_SIZE_IN_BYTS);                 	                                     			
			

        TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_DESFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
	
		#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
	    	return TESTFAIL;
	    #endif  
    
	} /* End of TST_ATP_EncryptDecryptFlag*/ 
	                                
  	return TST_TestStatus;
 }/* End of CRYS_ATP_TST_FLOW_DESSTATIC_Acceptance_WithAdditionalEngines*/ 


/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_DESSTATIC_Cyclic_Acceptance
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for static cyclic DES FLOW API's
 
 * Algorithm:     
 *	This function initializes DES flow with static type and other parameters,
 * and executes several times flow on the same input buffer. It goes throw all test
 * vectors and check cyclic acceptance.
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_DESSTATIC_Cyclic_Acceptance ( void ) 
{

	/********* TST Variables Definitions ************/
	int                     	 TST_index;
	CRYSError_t             	 TST_TestStatus = TESTPASS;
	char                     	 TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_DESSTATIC_Cyclic_Acceptance"; 
	

	/* initialize input and output buffers */
	TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;
	
	
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n"));
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_DESSTATIC_Cyclic_Acceptance Tests ********************* \n"));      
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n \n"));


	/* Go throw all test vectors and check cyclic acceptance */
	/*************************************************************************/
	for (TST_index =0 ; TST_index < TST_NumberOf_DESFLOW_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
    {
     	
     	TST_TestStatus = CRYS_ATP_TST_FLOW_DESSTATIC_ONETEST_Cyclic_Acceptance(TST_index, TST_MasterTestName);
     	
#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
		    return TESTFAIL;
#endif  		                                                               

	                            
	}/* End of TST_index */     
    
  	return TST_TestStatus;
 }/* End of CRYS_ATP_TST_FLOW_DESSTATIC_Acceptance_WithAdditionalEngines*/ 

 
#endif
